rm(list = ls())
library(shiny)
library(ggplot2)

#encapsulating the plot in a function
plotFunction <- function(input){
  
  set.seed(12345)
  
  numGenes <- 1000;
  numSamples <- 20;
  outcome <- c(rep(1, 10), rep(0, 10));
  dataset <- matrix(rnorm(numGenes * numSamples), 
                    numGenes, numSamples);
  dataset <- dataset + abs(min(dataset)) + 1;
  rownames(dataset) <- paste('S', 1:numGenes, sep = '')
  colnames(dataset) <- paste('G', 1:numSamples, sep = '')
  
  #significance
  pvalues <- apply(dataset, 1, 
                   function(x){t.test(x[outcome == 1], x[outcome == 0])$p.value})
  logPvalues <- -1 * log10(pvalues)
  logFoldChanges <- apply(dataset, 1, 
                          function(x){log2(mean(x[outcome == 1]) / mean(x[outcome == 0]))})
  significant <- logPvalues >= -1 * log10(input$threshold)
  significant[significant == TRUE] <- 'significant'
  significant[significant == 'FALSE'] <- 'non-significant'
  
  #gene characteristics
  pathway <- c(rep('Wnt signaling', numGenes/2), rep('MAPK signaling', numGenes/2));
  transcriptLength <- 10 * runif(numGenes);
  
  #storing all info in a dataset
  geneNames <- rownames(dataset)
  toPlot <- data.frame(geneNames,
                       logFoldChanges, 
                       logPvalues,
                       significant,
                       pathway,
                       transcriptLength)
  
  #volcano plot
  p <- ggplot() +
    geom_point(data = toPlot, 
               mapping = aes(x = logFoldChanges, 
                             y = logPvalues, 
                             color = significant,
                             shape = pathway,
                             size = transcriptLength),
               alpha = 0.7) +
    ggtitle('Volcano plot') +
    xlab('Log2 Fold Changes') +
    ylab('Log10 p-values') +
    scale_color_manual(values = c('blue', 'red')) +
    scale_size_continuous(name = 'transcript length') +
    theme(legend.position = ifelse(input$legendPresent == 'yes', 'right', 'none'))
  
  p
  
}

#user interface
ui <- fluidPage(

  sliderInput(inputId = 'threshold', label = 'FDR threshold', 
            min = 0, max = 1, value = 0.05),

  radioButtons(inputId = 'legendPresent', label = 'Plot legend', 
             choices = list('yes', 'no'), inline = TRUE),

  plotOutput(outputId = 'p')
  
)

#server side function
server <- function(input, output){
  
  #making the plot reactive
  p <- reactive(plotFunction(input));
  
  #preparing the plot
  output$p <- renderPlot(p(), width = 400);
  
}

#launching the app
shinyApp(ui = ui, server = server, 
         options = list(port = 8080, launch.browser = TRUE))
